(provide 'arpeggio.scm)

(my-require 'mylint.scm)
(my-require 'notes.scm)

;;
;; Note: Should be possible to store arpeggio-settings, and name them.
;;

'(define-struct pitch2
  :start
  :end
  :start-value
  :end-value
  :logtype)

(define-struct arpeggio
  :start
  :end
  
  ;; Values to arpeggiate. Each value is added to the currently playing note.
  :start-vals

  ;; E.g. for a C note where 'start-vals' is '(2 4), :num-notes-per-line==3 causes the three notes "C,D,E" to be played on a line (and not the 9 notes "C,D,E,C,D,E,C,D,E").
  ;; A value of 0 (or #f), marks that the current arpeggiation for the track is ending here. (must be used to stop an arpeggiator before the end-of-block, or it is not replaced by a new arpeggiator)
  :num-notes-per-line

  ;; 'end-vals' is defined if we want to pitch-glide the arpeggiating notes to other values during the span of this arpeggio.
  :end-vals #f

  ;; Applies random octave spread to the values.
  ;;
  ;; E.g. A value of 50 means that approx. 50% of the values will be at least one octave higher or lower, 25% of those again will be two octaves higher or lower, 12.5% of those again  will be three octaves higher or lower, etc.
  ;;
  ;; E.g. A value of 75 means that approx. 70% of the values will be at least one octave higher or lower, 35.% of those again will be two octaves higher or lower, etc.
  ;;
  ;; E.g. A value of 200 means that 200% of the values will be two octave higher or lower, 50% of those again will be three octaves higher or lower, etc.
  :randomize-octave-spread-factor 0
  
  ;; If end-vals is defined, we disable microtonality for it by setting this one to #f.
  :disable-cents-for-vals #f

  ;; All cents are disabled if this one is defined, even pitch-glides
  :disable-all-cents #f

  ;; If the note is pitch-gliding, disable pitch-gliding while doing arpeggio. (instead we get some kind of glide through the arpeggio, which might sound better)
  :disable-pitch-glide #f
  
  ;; Duration of individual notes (optional)
  :start-duration-weights #f
  :end-duration-weights #f ;; For gradually changing weights. Can be empty.

  ;; dB of individual notes (optional)
  :start-dbs #f
  :end-dbs #f ;; For gradually changing volume. Can be empty.
  )



(merge '((0 64 linear) (1 65 linear))
       '((0 0) (0.5 8)
         (1 0) (1.5 8)))
->
'((0 64 linear) (0.5 (+ 64.5 8) linear) (1.0 (+ 65 8) linear)
  (1 65 linear) (1.5 (+ 65 8) linear))

(pretty-print (lists->fxnodes `((0 64 0)
                                (0.5 ,(+ 64.5 8) 0)
                                (1.0 ,(+ 65 8) 0)
                                (1 65 0) (1.5 ,(+ 65 8) 0))))

(pretty-print (merge-fx-nodes (lists->fxnodes '((1 64 0)
                                                (6 65 0)))
                              (lists->fxnodes `((0 0 ,*logtype-hold*) (0.5 8 ,*logtype-hold*)
                                                (2 0 ,*logtype-hold*) (2.5 8 ,*logtype-hold*)))))


(define (split-pitches pitches pos kont)
  ...)

(define (split-pitches2 pitches start end kont)
  (split-pitches pitches start
                 (lambda (before mid-and-after)
                   (split-pitches mid-and-after end
                                  (lambda (mid after)
                                    (kont before mid after))))))

(coin)

(define (legalize-note-and-randomize-octave-spread factor note)
  (let loop ((note (if (<= factor 0)
                       note
                       (let ((add-func (if (> (random 100)
                                                 50)
                                           +
                                           -)))
                         (let loop ((factor factor)
                                    (note note))
                           (if (< (random 100)
                                  factor)
                               (loop (/ factor 2)
                                     (add-func note 12))
                               note))))))
    (cond ((<= note 0)
           (loop (+ note 12)))
          ((>= note 127)
           (loop (- note 12)))
          (else
           note))))

#!!
(legalize-note-and-randomize-octave-spread 200 64)
!##
 
;; 'apply-arpeggio-to-pitches-1' returns a list of pitches as a result of applying "arpeggio" to "pitch".
;;
;; The returned list has equal duration as 'pitch' [i.e. (- (next-pitch :place) (pitch :place))].
;; (except that the last pitch-node is not included.)
;;
;; The following four condition must always be true:
;; 1. (arpeggio :start) <= (pitch :place)
;; 2. (arpeggio :end) >= (next-pitch :place)
;; 3. start-place >= (pitch :place)
;; 4. start-place <= (next-pitch :place)
;; (i.e. 'pitch' must always be surrounded by 'arpeggio')
;;
(define (apply-arpeggio-to-pitches-1 note-start pitch next-pitch arpeggio start-place end-place)
  (define pitch-place-start (+ note-start (pitch :place)))
  (define pitch-place-end (+ note-start (next-pitch :place)))
  (define pitch-duration (- pitch-place-end pitch-place-start))

  (assert (<= (arpeggio :start) (pitch :place))) ;; 1.
  (assert (>= (arpeggio :end) (next-pitch :place))) ;; 2.
  (assert (>= start-place pitch-place-start))
  (assert (<= start-place pitch-place-end))
  (assert (>= start-place note-start))
  
  (define arpeggio-start (arpeggio :start))
  (define arpeggio-end (arpeggio :end))
  (define arpeggio-duration (- arpeggio-end arpeggio-start))
  
  (define num-vals (length (arpeggio :start-vals)))
  (define num-notes-per-line (arpeggio :num-notes-per-line))
  (define duration-between-notes (/ 1 num-notes-per-line))
  (define vals-start (list->vector (arpeggio :start-vals)))
  (define vals-end (and (arpeggio :end-vals) (list->vector (arpeggio :end-vals))))
  
  (assert (or (not vals-end) (= (vector-length vals-start) (vector-length vals-end))))
  
  (define pitch-value-start (pitch :value))
  (define pitch-value-end (next-pitch :value))
  
  (define is-gliding-pitch (and (= (pitch :logtype) *logtype-linear*)
                                (> (abs (- pitch-value-start pitch-value-end))
                                   0.001)))

  (define logtype (if (arpeggio :disable-pitch-glide)
                      *logtype-hold*
                      (pitch :logtype)))
  
  ;;(c-display "VALS-end:" vals-end)
  
  (let loop ((place start-place)
             (index 0))
    (if (>= place end-place)
        ;;(list (make-pitch :place place-end
        ;;                  :value (next-pitch :value)
        ;;                  :logtype (next-pitch :value)))
        '()
        (cons (make-pitch :place place
                          :value (let ((value (+ (if (not is-gliding-pitch)
                                                     pitch-value-start
                                                     (scale place
                                                            pitch-place-start pitch-place-end
                                                            pitch-value-start pitch-value-end))
                                                 (let ((arpeggio-val (if (not vals-end)
                                                                         (vector-ref vals-start index)
                                                                         (scale place
                                                                                arpeggio-start arpeggio-end
                                                                                (vector-ref vals-start index) (vector-ref vals-end index)))))
                                                   (if (and (not (arpeggio :disable-all-cents))
                                                            (arpeggio :disable-cents-for-vals))
                                                       (round arpeggio-val)
                                                       arpeggio-val)))))
                                   (legalize-note-and-randomize-octave-spread (arpeggio :randomize-octave-spread-factor)
                                                                              (if (arpeggio :disable-all-cents)
                                                                                  (round value)
                                                                                  value)))
                          :logtype logtype)
              (loop (+ place duration-between-notes)
                    (let ((next-index (+ index 1)))
                      (if (= next-index num-vals)
                          0
                          next-index)))))))

#!!
(pretty-print (apply-arpeggio-to-pitches-1 1
                                           (make-pitch :place 2.0
                                                       :value 64.0
                                                       :logtype *logtype-linear*)
                                           (make-pitch :place 4
                                                       :value 64
                                                       :logtype *logtype-linear*)
                                           (make-arpeggio :start 0
                                                          :end 5
                                                          :start-vals '(0 5)
                                                          :num-notes-per-line 12
                                                          :end-vals '(50 60)
                                                          :randomize-octave-spread-factor 200
                                                          :disable-all-cents #t)
                                           3
                                           4))
(pretty-print (apply-arpeggio-to-pitches-1 (make-pitch :place 5.0
                                                       :value 64.0
                                                       :logtype *logtype-linear*)
                                           (make-pitch :place 6
                                                       :value 65
                                                       :logtype *logtype-linear*)
                                           (make-arpeggio :start 0
                                                          :end 5
                                                          :start-vals '(0 5 8 9 -100)
                                                          :num-notes-per-line 8
                                                          )
                                           0))
!!#

;; Returns new pitches for the note with arpeggios applied.
(define (apply-arpeggios-to-pitches note arpeggios)
  (let loop ((pitches (note :pitches)))
    (cond ((null? pitches)
           '())
          ((null? arpeggios)
           pitches)
          (else
           (define arpeggio (car arpeggios))
           (define start (arpeggio :start))
           (define end (arpeggio :end))
           (define delta-time (- start (note :place)))
           (split-pitches2 pitches start end
                           (lambda (pitches-before pitches pitches-after)
                             (flatten (list pitches-before
                                            (let loop2 ((pitches pitches))
                                              (define pitch (cl-car pitches))
                                              (define next-pitch (cl-cadr pitches))
                                              (if pitch
                                                  (assert next-pitch)) ;; Pretty sure...
                                              (if (and pitch next-pitch)
                                                  (cons (apply-arpeggio-to-pitches-1 pitch
                                                                                     next-pitch
                                                                                     arpeggio
                                                                                     (note :place))
                                                        (loop2 (cdr pitches)))
                                                  '()))
                                            (loop pitches-after
                                                  (cdr arpeggios)
                                                  block-end)))))))))
                

(apply-arpeggios-to-pitches `((make-pitch :place 0
                                          :value 64
                                          :logtype ,*logtype-linear*)
                              (make-pitch :place 1
                                          :value 65
                                          :logtype ,*logtype-linear*))
                            (list (make-arpeggio :start 0
                                                 :end 2
                                                 :start-vals '(0 8)
                                                 :num-notes-per-line 6
                                                 )
                                  (make-arpeggio :start 10
                                                 :end 11
                                                 :start-vals '(0 8)
                                                 :num-notes-per-line 8
                                                 )))

->
'((0 64 linear) (0.5 (+ 64.5 8) linear) (1.0 (+ 65 8) linear)
  (1 65 linear) (1.5 (+ 65 8) linear))

(make-hash* :line 5 :hepp 9)
